const db = require("../../config/db");

class PrecosRepository {

  #db = db;

  /* ================= CREATE ================= */
  async create(data) {
    try {
      await this.#db.query(
        `INSERT INTO precos (id_produto, id_unidade, preco)
         VALUES (?, ?, ?)`,
        [data.id_produto, data.id_unidade, data.preco]
      );

      return {
        success: true,
        message: "Preço cadastrado com sucesso!",
        code: 201
      };
    } catch (error) {
      return {
        error: true,
        message: "Erro ao cadastrar preço",
        code: 500,
        detail: error
      };
    }
  }

  /* ================= LISTAR TODOS ================= */
  async getPrecos() {
    try {
      const precos = await this.#db.query(
        `SELECT 
            p.id,
            p.preco,
            pr.nome AS produto,
            u.nome AS unidade
         FROM precos p
         INNER JOIN produtos pr ON pr.id = p.id_produto
         INNER JOIN unidades u ON u.id = p.id_unidade`
      );

      return precos;
    } catch (error) {
      return {
        error: true,
        message: "Erro ao buscar preços",
        code: 500,
        detail: error
      };
    }
  }

  /* ================= BUSCAR UM ================= */
  async getUmPreco(id) {
    try {
      const precos = await this.#db.query(
        `SELECT 
            p.id,
            p.preco,
            p.id_produto,
            p.id_unidade,
            pr.nome AS produto,
            u.nome AS unidade
         FROM precos p
         INNER JOIN produtos pr ON pr.id = p.id_produto
         INNER JOIN unidades u ON u.id = p.id_unidade
         WHERE p.id = ?`,
        [id]
      );

      if (!precos.length) {
        return {
          error: true,
          message: "Preço não encontrado",
          code: 404
        };
      }

      return precos[0];
    } catch (error) {
      return {
        error: true,
        message: "Erro ao buscar preço",
        code: 500,
        detail: error
      };
    }
  }

  /**
   * Atualiza um preço
   * @param {Object} data
   */
  async update(data) {
    try {
      await this.#db.query(
        `UPDATE precos 
         SET id_produto = ?, id_unidade = ?, preco = ?
         WHERE id = ?`,
        [
          data.id_produto,
          data.id_unidade,
          data.preco,
          data.id
        ]
      );

      return {
        success: true,
        message: "Preço atualizado com sucesso!",
        code: 200
      };
    } catch (error) {
      return {
        error: true,
        message: "Erro ao atualizar preço",
        code: 500,
        detail: error
      };
    }
  }

  /**
   * Deleta um preço
   * @param {Int} id
   */
  async delete(id) {
    try {
      await this.#db.query(
        `DELETE FROM precos WHERE id = ?`,
        [id]
      );

      return {
        success: true,
        message: "Preço deletado com sucesso!",
        code: 200
      };
    } catch (error) {
      return {
        error: true,
        message: "Erro ao deletar preço",
        code: 500,
        detail: error
      };
    }
  }
}

module.exports = new PrecosRepository();
