const db = require("../../config/db");
const bcrypt = require("bcrypt");

class ContatosRepository {

  #db = db;

  /* ================= CREATE ================= */
  async create(data) {
    try {
      const dataAtual = new Date();

      await this.#db.query(
        `INSERT INTO contatos (
          razao_social, documento, celular, email, role,
          cep, logradouro, complemento, unidade, bairro, localidade,
          uf, estado, anotacoes, data_criado,
          notificacao_email, notificacao_whats, cadastro
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          data.razao_social,
          data.documento,
          data.celular,
          data.email,
          data.role || "USER",

          data.cep || "",
          data.logradouro || "",
          data.complemento || "",
          data.unidade || "",
          data.bairro || "",
          data.localidade || "",
          data.uf || "",
          data.estado || "",

          data.anotacoes || "",
          dataAtual,

          data.notificacao_email || 0,
          data.notificacao_whats || 0,
          data.cadastro || null
        ]
      );

      return {
        success: true,
        message: "Usuário cadastrado com sucesso!",
        code: 201
      };
    } catch (error) {
      return {
        error: true,
        message: "Erro ao cadastrar usuário",
        code: 500,
        detail: error
      };
    }
  }

  /* ================= LISTAR TODOS ================= */
  async getUsers() {
    try {
      const users = await this.#db.query(
        `SELECT 
          id,
          razao_social,
          documento,
          celular,
          email,
          role,
          DATE_FORMAT(data_criado, '%d/%m/%Y %H:%i') AS data_criado,
          notificacao_email,
          notificacao_whats,
          cadastro
        FROM user`
      );

      return users;
    } catch (error) {
      return {
        error: true,
        message: "Erro ao buscar usuários",
        code: 500,
        detail: error
      };
    }
  }

  /* ================= BUSCAR UM ================= */
  async getUserById(id) {
    try {
      const users = await this.#db.query(
        `SELECT 
          id,
          razao_social,
          documento,
          celular,
          email,
          role,
          cep,
          logradouro,
          complemento,
          unidade,
          bairro,
          localidade,
          uf,
          estado,
          anotacoes,
          DATE_FORMAT(data_criado, '%d/%m/%Y %H:%i') AS data_criado,
          notificacao_email,
          notificacao_whats,
          cadastro
        FROM contatos
        WHERE id = ?`,
        [id]
      );

      if (!users.length) {
        return {
          error: true,
          message: "Usuário não encontrado",
          code: 404
        };
      }

      return users[0];
    } catch (error) {
      return {
        error: true,
        message: "Erro ao buscar usuário",
        code: 500,
        detail: error
      };
    }
  }

  /* ================= UPDATE ================= */
  async update(data) {
    try {
      await this.#db.query(
        `UPDATE contatos SET
          razao_social = ?,
          celular = ?,
          email = ?,
          cep = ?,
          logradouro = ?,
          complemento = ?,
          unidade = ?,
          bairro = ?,
          localidade = ?,
          uf = ?,
          estado = ?,
          anotacoes = ?,
          notificacao_email = ?,
          notificacao_whats = ?
        WHERE id = ?`,
        [
          data.razao_social,
          data.celular,
          data.email,
          data.cep,
          data.logradouro,
          data.complemento,
          data.unidade,
          data.bairro,
          data.localidade,
          data.uf,
          data.estado,
          data.anotacoes,
          data.notificacao_email,
          data.notificacao_whats,
          data.id
        ]
      );

      return {
        success: true,
        message: "Usuário atualizado com sucesso!",
        code: 200
      };
    } catch (error) {
      return {
        error: true,
        message: "Erro ao atualizar usuário",
        code: 500,
        detail: error
      };
    }
  }

  /* ================= UPDATE SENHA ================= */
  async updatePassword(id, novaSenha) {
    try {
      const hashedPassword = await bcrypt.hash(novaSenha, 10);

      await this.#db.query(
        `UPDATE user SET senha = ? WHERE id = ?`,
        [hashedPassword, id]
      );

      return {
        success: true,
        message: "Senha atualizada com sucesso!",
        code: 200
      };
    } catch (error) {
      return {
        error: true,
        message: "Erro ao atualizar senha",
        code: 500,
        detail: error
      };
    }
  }

  /* ================= DELETE ================= */
  async delete(id) {
    try {
      await this.#db.query(
        `DELETE FROM contatos WHERE id = ?`,
        [id]
      );

      return {
        success: true,
        message: "Usuário deletado com sucesso!",
        code: 200
      };
    } catch (error) {
      return {
        error: true,
        message: "Erro ao deletar usuário",
        code: 500,
        detail: error
      };
    }
  }
}

module.exports = new ContatosRepository();
